// MarkerImporter.jsx - ExtendScript for Premiere Pro

function importMarkers(markersJSON) {
    try {
        // Parse the markers data
        var markers = JSON.parse(markersJSON);
        
        // Get the active sequence
        var sequence = app.project.activeSequence;
        
        if (!sequence) {
            return "ERROR: Keine aktive Sequenz gefunden. Bitte öffne eine Sequenz.";
        }
        
        var successCount = 0;
        var errorCount = 0;
        var errorDetails = [];
        
        // Import each marker
        for (var i = 0; i < markers.length; i++) {
            try {
                var markerData = markers[i];
                
                // Convert In timecode string to seconds
                var inTimeSeconds = timecodeToSeconds(markerData.inPoint, sequence);
                
                if (inTimeSeconds === null || inTimeSeconds < 0) {
                    errorCount++;
                    errorDetails.push("Zeile " + (i+2) + " (" + markerData.name + "): Ungültiger Timecode " + markerData.inPoint);
                    continue;
                }
                
                // Create marker at In point (Premiere expects seconds for createMarker)
                var marker = sequence.markers.createMarker(inTimeSeconds);
                
                if (!marker) {
                    errorCount++;
                    errorDetails.push("Marker konnte nicht erstellt werden: " + markerData.name);
                    continue;
                }
                
                // Set marker properties
                marker.name = markerData.name || "Marker " + (i + 1);
                marker.comments = markerData.description || "";
                
                // Set marker type
                var mType = markerData.markerType ? markerData.markerType.toLowerCase() : "comment";
                marker.type = "Comment"; // Default to Comment for now
                
                // Set duration/end time if Out point is different from In point
                if (markerData.outPoint && markerData.outPoint !== markerData.inPoint && markerData.outPoint !== "00:00:00:00") {
                    var outTimeSeconds = timecodeToSeconds(markerData.outPoint, sequence);
                    if (outTimeSeconds !== null && outTimeSeconds > inTimeSeconds) {
                        marker.end = outTimeSeconds;
                    }
                }
                
                // Set color based on hex value
                if (markerData.color) {
                    try {
                        var colorIndex = hexToPremiereColor(markerData.color);
                        marker.setColorByIndex(colorIndex);
                    } catch (colorError) {
                        // Color setting failed, but marker is still created
                    }
                }
                
                successCount++;
                
            } catch (e) {
                errorCount++;
                var errMsg = "Marker " + (i+2) + ": " + e.toString();
                errorDetails.push(errMsg);
            }
        }
        
        var result = "SUCCESS: " + successCount + " importiert";
        if (errorCount > 0) {
            result += ", " + errorCount + " Fehler";
        }
        
        return result;
        
    } catch (e) {
        return "ERROR: " + e.toString();
    }
}

function timecodeToSeconds(timecodeStr, sequence) {
    try {
        if (!timecodeStr || timecodeStr === '') {
            return null;
        }
        
        // Clean the timecode string - remove all whitespace
        timecodeStr = timecodeStr.replace(/[\s\t\r\n]/g, '');
        
        // Replace semicolons with colons
        timecodeStr = timecodeStr.replace(/;/g, ':');
        
        // Split by colon
        var parts = timecodeStr.split(':');
        
        if (parts.length !== 4) {
            return null;
        }
        
        var hours = parseInt(parts[0], 10);
        var minutes = parseInt(parts[1], 10);
        var seconds = parseInt(parts[2], 10);
        var frames = parseInt(parts[3], 10);
        
        if (isNaN(hours) || isNaN(minutes) || isNaN(seconds) || isNaN(frames)) {
            return null;
        }
        
        // Get framerate from sequence
        var framerate = parseFloat(sequence.framerate);
        
        if (isNaN(framerate) || framerate <= 0) {
            framerate = 25.0;
        }
        
        // Calculate total seconds
        var totalSeconds = (hours * 3600) + (minutes * 60) + seconds + (frames / framerate);
        
        return totalSeconds;
        
    } catch (e) {
        return null;
    }
}

function hexToPremiereColor(hexColor) {
    // Map hex colors to Premiere Pro marker color indices
    // Premiere has predefined colors: 0-15
    // Common colors based on the example CSV
    var colorMap = {
        "718637": 0,  // Green
        "D22C36": 1,  // Red
        "AF8BB1": 2,  // Purple/Lavender
        "E96F24": 3,  // Orange
        "D0A12B": 4,  // Yellow
        "428DFC": 5,  // Blue
        "19F4D6": 6,  // Turquoise/Cyan
        "1A1A1A": 7,  // Dark gray
        "FFFFFF": 8,  // White
        "00FF00": 0,  // Green fallback
        "FF0000": 1,  // Red fallback
        "0000FF": 5,  // Blue fallback
        "FFFF00": 4,  // Yellow fallback
        "FF8800": 3,  // Orange fallback
        "00FFFF": 6,  // Cyan fallback
        "FF00FF": 2   // Magenta fallback
    };
    
    // Remove # if present
    hexColor = hexColor.replace("#", "").toUpperCase();
    
    if (colorMap[hexColor] !== undefined) {
        return colorMap[hexColor];
    }
    
    // Return default color if not found
    return 0;
}



// Helper function to get sequence info (for debugging)
function getSequenceInfo() {
    try {
        var sequence = app.project.activeSequence;
        if (!sequence) {
            return "No active sequence";
        }
        
        return "Sequence: " + sequence.name + ", Framerate: " + sequence.framerate;
    } catch (e) {
        return "Error: " + e.toString();
    }
}
